/**
 * Glavna JavaScript datoteka za aplikaciju za detekciju dimenzija
 * Putanja na produkciji: /var/www/html/js/main.js
*/

// Globalne varijable
let currentFile = null;
let currentPdf = null;
let currentPage = 1;
let totalPages = 1;
let currentScale = 1.0;
let dimensions = [];
let isManualMode = false;
let categorizedDimensions = {
    dimension: [],
    thread: [],
    roughness: [],
    gdt: []
};

// Kategorije i podkategorije
const dimensionCategories = {
    dimension: {
        name: "Dimenzije",
        subtypes: [
            { value: "linear", label: "Linearne dimenzije" },
            { value: "diameter", label: "Promjeri (Ø)" },
            { value: "radius", label: "Radiusi (R)" },
            { value: "angle", label: "Kutevi" },
            { value: "tolerance", label: "Tolerancije" }
        ]
    },
    thread: {
        name: "Navoji",
        subtypes: [
            { value: "metric", label: "Metrički navoji" },
            { value: "imperial", label: "Inčni navoji" },
            { value: "other", label: "Ostali navoji" }
        ]
    },
    roughness: {
        name: "Hrapavost",
        subtypes: [
            { value: "ra", label: "Ra" },
            { value: "rz", label: "Rz" }
        ]
    },
    gdt: {
        name: "GD&T",
        subtypes: [
            { value: "flatness", label: "Ravnost" },
            { value: "straightness", label: "Pravost" },
            { value: "circularity", label: "Kružnost" },
            { value: "cylindricity", label: "Cilindričnost" },
            { value: "parallelism", label: "Paralelnost" },
            { value: "perpendicularity", label: "Okomitost" },
            { value: "angularity", label: "Kutnost" },
            { value: "concentricity", label: "Koncentričnost" },
            { value: "symmetry", label: "Simetrija" },
            { value: "position", label: "Pozicija" }
        ]
    }
};

// Regularni izrazi za prepoznavanje različitih formata dimenzija
const dimensionPatterns = {
    // Promjeri
    diameter: /Ø\s*\d+(\.\d+)?(\s*[a-zA-Z]\d+)?(\s*\(\+?[\d\.-]+\/?[+-]?[\d\.-]+\))?/g,
    
    // Navoji
    thread: /M\d+(\.\d+)?\s*[×x]\s*\d+(\.\d+)?(---\d+[a-zA-Z]\/\d+[a-zA-Z])?/g,
    
    // Hrapavosti
    roughness: /▽√?U?Ra\s*\d+(\.\d+)?/g,
    
    // GD&T
    gdt: /\[\s*[↗⟂◎⌭]\s*\|\s*[Ø]?\d+(\.\d+)?\s*\|\s*[A-Z]-[A-Z]\s*\]|\[\s*[↗⟂◎⌭]\s*\|\s*[Ø]?\d+(\.\d+)?\s*\]/g,
    
    // Opće dimenzije
    dimension: /\d+(\.\d+)?(\s*[±]\s*\d+(\.\d+)?)?/g
};

// Inicijalizacija nakon učitavanja DOM-a
document.addEventListener('DOMContentLoaded', function() {
    // Inicijalizacija kontrola
    initializeControls();
    
    // Postavljanje event listenera
    setupEventListeners();
    
    // Priprema kategorijskih tabova
    initializeCategoryTabs();
});

/**
 * Inicijalizacija kontrola na stranici
 */
function initializeControls() {
    // Inicijalizacija dropdown-a za podkategorije
    const dimensionTypeSelect = document.getElementById('dimensionType');
    const dimensionSubTypeSelect = document.getElementById('dimensionSubType');
    
    dimensionTypeSelect.addEventListener('change', function() {
        updateSubTypeOptions(this.value);
    });
    
    // Inicijalno popuni podkategorije
    updateSubTypeOptions(dimensionTypeSelect.value);
}

/**
 * Ažuriranje opcija za podkategorije
 * @param {string} category - Odabrana kategorija
 */
function updateSubTypeOptions(category) {
    const dimensionSubTypeSelect = document.getElementById('dimensionSubType');
    dimensionSubTypeSelect.innerHTML = '<option value="">-- Odaberite podkategoriju --</option>';
    
    if (dimensionCategories[category]) {
        dimensionCategories[category].subtypes.forEach(subtype => {
            const option = document.createElement('option');
            option.value = subtype.value;
            option.textContent = subtype.label;
            dimensionSubTypeSelect.appendChild(option);
        });
    }
}

/**
 * Postavljanje event listenera za kontrole
 */
function setupEventListeners() {
    // Upload formulara
    document.getElementById('uploadForm').addEventListener('submit', function(e) {
        e.preventDefault();
        handleFileUpload();
    });
    
    // Gumbi za akcije
    document.getElementById('detectDimensionsBtn').addEventListener('click', detectDimensions);
    document.getElementById('manualModeBtn').addEventListener('click', toggleManualMode);
    document.getElementById('exportBtn').addEventListener('click', exportToExcel);
    document.getElementById('resetBtn').addEventListener('click', resetDocument);
    
    // Zoom kontrole
    document.getElementById('zoomInBtn').addEventListener('click', zoomIn);
    document.getElementById('zoomOutBtn').addEventListener('click', zoomOut);
    document.getElementById('resetZoomBtn').addEventListener('click', resetZoom);
    
    // Modal za dodavanje dimenzije
    document.getElementById('saveDimensionBtn').addEventListener('click', saveDimension);
    
    // Dokument viewer (za ručno označavanje)
    document.getElementById('viewerContainer').addEventListener('click', function(e) {
        if (isManualMode) {
            handleViewerClick(e);
        }
    });
}

/**
 * Inicijalizacija kategorijskih tabova
 */
function initializeCategoryTabs() {
    // Ovdje možemo dodati eventualnu inicijalizaciju tabova
}

/**
 * Obrađuje upload datoteke
 */
function handleFileUpload() {
    const fileInput = document.getElementById('documentFile');
    const file = fileInput.files[0];
    
    if (!file) {
        showAlert('Molimo odaberite datoteku.', 'warning');
        return;
    }
    
    // Provjera tipa datoteke
    const allowedTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 'image/tiff', 'image/bmp'];
    if (!allowedTypes.includes(file.type)) {
        showAlert('Podržani formati: PDF, JPEG, PNG, TIFF i BMP.', 'warning');
        return;
    }
    
    // Prikaži loading indikator
    showLoadingIndicator(true);
    
    // Simuliramo upload na server (u pravoj implementaciji, ovo bi bio AJAX poziv)
    setTimeout(() => {
        currentFile = file;
        
        // Renderiranje datoteke ovisno o tipu
        if (file.type === 'application/pdf') {
            renderPdfDocument(file);
        } else {
            renderImageDocument(file);
        }
        
        // Omogući gumbe
        enableControls();
        
        showLoadingIndicator(false);
        showAlert('Dokument uspješno učitan!', 'success');
    }, 1000);
}

/**
 * Renderira PDF dokument
 * @param {File} file - PDF datoteka
 */
function renderPdfDocument(file) {
    const fileReader = new FileReader();
    
    fileReader.onload = function() {
        const typedArray = new Uint8Array(this.result);
        
        // Koristi pdf.js za prikaz PDF-a
        pdfjsLib.getDocument(typedArray).promise.then(function(pdf) {
            currentPdf = pdf;
            totalPages = pdf.numPages;
            
            renderPage(1);
            
            document.getElementById('noDocumentMessage').classList.add('d-none');
            document.getElementById('viewerContainer').classList.remove('d-none');
        });
    };
    
    fileReader.readAsArrayBuffer(file);
}

/**
 * Renderira stranicu PDF-a
 * @param {number} pageNumber - Broj stranice za renderiranje
 */
function renderPage(pageNumber) {
    currentPage = pageNumber;
    
    currentPdf.getPage(pageNumber).then(function(page) {
        const viewport = page.getViewport({ scale: currentScale });
        
        // Priprema canvas elementa
        const viewerContainer = document.getElementById('viewerContainer');
        viewerContainer.innerHTML = '';
        
        const canvas = document.createElement('canvas');
        viewerContainer.appendChild(canvas);
        
        const context = canvas.getContext('2d');
        canvas.height = viewport.height;
        canvas.width = viewport.width;
        
        // Renderiranje PDF-a na canvas
        const renderContext = {
            canvasContext: context,
            viewport: viewport
        };
        
        page.render(renderContext).promise.then(function() {
            // Ponovno iscrtaj markere dimenzija ako postoje
            renderDimensionMarkers();
        });
    });
}

/**
 * Renderira sliku (JPEG, PNG, itd.)
 * @param {File} file - Slika datoteka
 */
function renderImageDocument(file) {
    const fileReader = new FileReader();
    
    fileReader.onload = function() {
        // Priprema kontejnera
        document.getElementById('noDocumentMessage').classList.add('d-none');
        document.getElementById('viewerContainer').classList.remove('d-none');
        
        const viewerContainer = document.getElementById('viewerContainer');
        viewerContainer.innerHTML = '';
        
        // Stvori sliku
        const img = document.createElement('img');
        img.src = this.result;
        img.style.maxWidth = '100%';
        img.style.transform = `scale(${currentScale})`;
        img.style.transformOrigin = '0 0';
        
        viewerContainer.appendChild(img);
        
        // Postavi PDF na null jer se radi o slici
        currentPdf = null;
        totalPages = 1;
        currentPage = 1;
    };
    
    fileReader.readAsDataURL(file);
}

/**
 * Automatska detekcija dimenzija
 */
function detectDimensions() {
    // Prikaži loading indikator
    showLoadingIndicator(true);
    
    // U realnoj implementaciji, ovo bi bio AJAX poziv na server koji poziva Python skriptu
    // Simuliramo detekciju s timeout-om
    setTimeout(() => {
        // Generiramo neke dummy dimenzije
        dimensions = [];
        let count = 0;
        
        // Simulirane dimenzije za demonstraciju
        const sampleDimensions = [
            { text: "Ø28 k6 (+0.015/+0.002)", type: "dimension", subtype: "diameter" },
            { text: "Ø13", type: "dimension", subtype: "diameter" },
            { text: "Ø44.2 h7 (0/-0.025)", type: "dimension", subtype: "diameter" },
            { text: "15.46", type: "dimension", subtype: "linear" },
            { text: "20.55", type: "dimension", subtype: "linear" },
            { text: "M10×1.5---6g/6H", type: "thread", subtype: "metric" },
            { text: "▽√URa 3.2", type: "roughness", subtype: "ra" },
            { text: "[⟂|0.05|A-B]", type: "gdt", subtype: "perpendicularity" }
        ];
        
        // Generiramo random pozicije za demonstraciju
        for (let i = 0; i < sampleDimensions.length; i++) {
            count++;
            
            const dim = {
                number: count,
                text: sampleDimensions[i].text,
                type: sampleDimensions[i].type,
                subtype: sampleDimensions[i].subtype,
                x: Math.floor(Math.random() * 600) + 50,
                y: Math.floor(Math.random() * 400) + 50,
                page: 1
            };
            
            dimensions.push(dim);
        }
        
        // Kategoriziramo dimenzije
        categorizeDimensions();
        
        // Prikaz dimenzija
        renderDimensionsList();
        renderDimensionMarkers();
        
        showLoadingIndicator(false);
        showAlert('Automatska detekcija završena. Pronađeno ' + dimensions.length + ' dimenzija.', 'success');
    }, 2000);
}

/**
 * Kategorizira dimenzije prema tipu
 */
function categorizeDimensions() {
    // Resetiraj kategorije
    categorizedDimensions = {
        dimension: [],
        thread: [],
        roughness: [],
        gdt: []
    };
    
    // Kategoriziraj dimenzije
    dimensions.forEach(dim => {
        if (categorizedDimensions[dim.type]) {
            categorizedDimensions[dim.type].push(dim);
        }
    });
    
    // Ažuriraj prikaz kategorija
    updateCategoryTabs();
}

/**
 * Ažurira tabove s kategoriziranim dimenzijama
 */
function updateCategoryTabs() {
    // Ažuriraj svaki tab
    for (const category in categorizedDimensions) {
        const tabContent = document.getElementById(category);
        if (!tabContent) continue;
        
        if (categorizedDimensions[category].length === 0) {
            tabContent.innerHTML = '<p class="text-center text-muted">Nema prepoznatih elemenata</p>';
            continue;
        }
        
        let html = '<ul class="list-group">';
        categorizedDimensions[category].forEach(dim => {
            html += `
                <li class="list-group-item d-flex justify-content-between align-items-center">
                    <span class="dimension-number">${dim.number}</span>
                    <span class="dimension-text">${dim.text}</span>
                </li>
            `;
        });
        html += '</ul>';
        
        tabContent.innerHTML = html;
    }
}

/**
 * Renderira listu dimenzija
 */
function renderDimensionsList() {
    const dimensionsList = document.getElementById('dimensionsList');
    
    if (dimensions.length === 0) {
        dimensionsList.innerHTML = '<div class="list-group-item text-center p-3" id="noDimensionsMessage"><p class="mb-0">Nema prepoznatih dimenzija</p></div>';
        return;
    }
    
    // Ukloni poruku da nema dimenzija
    const noDimensionsMessage = document.getElementById('noDimensionsMessage');
    if (noDimensionsMessage) {
        noDimensionsMessage.remove();
    }
    
    // Generiranje HTML-a za listu dimenzija
    let html = '';
    dimensions.forEach(dim => {
        html += `
            <div class="dimension-item" data-dimension-id="${dim.number}">
                <div class="d-flex align-items-center">
                    <span class="dimension-number">${dim.number}</span>
                    <span class="dimension-text">${dim.text}</span>
                </div>
                <div class="dimension-actions">
                    <button class="btn btn-sm btn-outline-primary edit-dimension-btn" data-dimension-id="${dim.number}">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-outline-danger delete-dimension-btn" data-dimension-id="${dim.number}">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        `;
    });
    
    dimensionsList.innerHTML = html;
    
    // Dodaj event listenere za gumbe za uređivanje i brisanje
    document.querySelectorAll('.edit-dimension-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const dimensionId = parseInt(this.getAttribute('data-dimension-id'));
            editDimension(dimensionId);
        });
    });
    
    document.querySelectorAll('.delete-dimension-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const dimensionId = parseInt(this.getAttribute('data-dimension-id'));
            deleteDimension(dimensionId);
        });
    });
}

/**
 * Renderira markere za dimenzije na dokumentu
 */
function renderDimensionMarkers() {
    // Ukloni sve postojeće markere
    document.querySelectorAll('.dimension-marker').forEach(marker => {
        marker.remove();
    });
    
    // Dodaj markere za trenutnu stranicu
    const viewerContainer = document.getElementById('viewerContainer');
    
    dimensions.filter(dim => dim.page === currentPage).forEach(dim => {
        const marker = document.createElement('div');
        marker.className = 'dimension-marker';
        marker.textContent = dim.number;
        marker.style.left = dim.x + 'px';
        marker.style.top = dim.y + 'px';
        marker.setAttribute('data-dimension-id', dim.number);
        
        viewerContainer.appendChild(marker);
    });
}

/**
 * Uključuje/isključuje ručni način označavanja
 */
function toggleManualMode() {
    isManualMode = !isManualMode;
    const manualModeBtn = document.getElementById('manualModeBtn');
    const documentViewer = document.querySelector('.document-viewer');
    
    if (isManualMode) {
        manualModeBtn.classList.remove('btn-info');
        manualModeBtn.classList.add('btn-warning');
        manualModeBtn.innerHTML = '<i class="fas fa-times me-2"></i>Izađi iz ručnog načina';
        documentViewer.classList.add('manual-mode-active');
        showAlert('Ručni način aktiviran. Kliknite na dokument za dodavanje dimenzije.', 'info');
    } else {
        manualModeBtn.classList.remove('btn-warning');
        manualModeBtn.classList.add('btn-info');
        manualModeBtn.innerHTML = '<i class="fas fa-pencil-alt me-2"></i>Ručno označavanje';
        documentViewer.classList.remove('manual-mode-active');
        showAlert('Ručni način deaktiviran.', 'info');
    }
}

/**
 * Obrađuje klik na viewer u ručnom načinu
 * @param {Event} e - Event objekt
 */
function handleViewerClick(e) {
    // Izračunaj poziciju klika u odnosu na viewer
    const viewerContainer = document.getElementById('viewerContainer');
    const rect = viewerContainer.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const y = e.clientY - rect.top;
    
    // Postavi poziciju u modal
    document.getElementById('dimensionX').value = x;
    document.getElementById('dimensionY').value = y;
    document.getElementById('dimensionPage').value = currentPage;
    
    // Prikaži modal za unos dimenzije
    const modal = new bootstrap.Modal(document.getElementById('addDimensionModal'));
    modal.show();
}

/**
 * Sprema novu dimenziju iz modala
 */
function saveDimension() {
    const dimensionText = document.getElementById('dimensionText').value;
    const dimensionType = document.getElementById('dimensionType').value;
    const dimensionSubType = document.getElementById('dimensionSubType').value;
    const x = parseFloat(document.getElementById('dimensionX').value);
    const y = parseFloat(document.getElementById('dimensionY').value);
    const page = parseInt(document.getElementById('dimensionPage').value);
    
    if (!dimensionText) {
        showAlert('Molimo unesite tekst dimenzije.', 'warning');
        return;
    }
    
    // Dodaj novu dimenziju
    const newDimension = {
        number: dimensions.length > 0 ? Math.max(...dimensions.map(d => d.number)) + 1 : 1,
        text: dimensionText,
        type: dimensionType,
        subtype: dimensionSubType,
        x: x,
        y: y,
        page: page
    };
    
    dimensions.push(newDimension);
    
    // Ažuriraj prikaz
    categorizeDimensions();
    renderDimensionsList();
    renderDimensionMarkers();
    
    // Zatvori modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('addDimensionModal'));
    modal.hide();
    
    // Resetiraj formu
    document.getElementById('dimensionForm').reset();
    
    showAlert('Dimenzija uspješno dodana.', 'success');
}

/**
 * Uređuje postojeću dimenziju
 * @param {number} dimensionId - ID dimenzije za uređivanje
 */
function editDimension(dimensionId) {
    const dimension = dimensions.find(d => d.number === dimensionId);
    if (!dimension) return;
    
    // Popuni modal s podacima
    document.getElementById('dimensionText').value = dimension.text;
    document.getElementById('dimensionType').value = dimension.type;
    updateSubTypeOptions(dimension.type);
    document.getElementById('dimensionSubType').value = dimension.subtype || '';
    document.getElementById('dimensionX').value = dimension.x;
    document.getElementById('dimensionY').value = dimension.y;
    document.getElementById('dimensionPage').value = dimension.page;
    
    // Promijeni akciju modala
    const saveDimensionBtn = document.getElementById('saveDimensionBtn');
    saveDimensionBtn.textContent = 'Ažuriraj';
    
    // Promijeni ponašanje gumba za spremanje
    const originalSaveFunction = saveDimensionBtn.onclick;
    saveDimensionBtn.onclick = function() {
        // Ažuriraj dimenziju
        dimension.text = document.getElementById('dimensionText').value;
        dimension.type = document.getElementById('dimensionType').value;
        dimension.subtype = document.getElementById('dimensionSubType').value;
        dimension.x = parseFloat(document.getElementById('dimensionX').value);
        dimension.y = parseFloat(document.getElementById('dimensionY').value);
        dimension.page = parseInt(document.getElementById('dimensionPage').value);
        
        // Ažuriraj prikaz
        categorizeDimensions();
        renderDimensionsList();
        renderDimensionMarkers();
        
        // Zatvori modal
        const modal = bootstrap.Modal.getInstance(document.getElementById('addDimensionModal'));
        modal.hide();
        
        // Resetiraj formu i vrati originalni handler
        document.getElementById('dimensionForm').reset();
        saveDimensionBtn.onclick = originalSaveFunction;
        saveDimensionBtn.textContent = 'Spremi';
        
        showAlert('Dimenzija uspješno ažurirana.', 'success');
    };
    
    // Prikaži modal
    const modal = new bootstrap.Modal(document.getElementById('addDimensionModal'));
    modal.show();
}

/**
 * Briše dimenziju
 * @param {number} dimensionId - ID dimenzije za brisanje
 */
function deleteDimension(dimensionId) {
    if (confirm('Jeste li sigurni da želite obrisati ovu dimenziju?')) {
        // Pronađi indeks dimenzije
        const index = dimensions.findIndex(d => d.number === dimensionId);
        if (index !== -1) {
            // Ukloni dimenziju
            dimensions.splice(index, 1);
            
            // Ažuriraj prikaz
            categorizeDimensions();
            renderDimensionsList();
            renderDimensionMarkers();
            
            showAlert('Dimenzija uspješno obrisana.', 'success');
        }
    }
}

/**
 * Izvozi dimenzije u Excel
 */
function exportToExcel() {
    if (dimensions.length === 0) {
        showAlert('Nema dimenzija za izvoz.', 'warning');
        return;
    }
    
    showLoadingIndicator(true);
    
    // U realnoj implementaciji, ovo bi bio AJAX poziv na server
    // Simuliramo izvoz s timeout-om
    setTimeout(() => {
        showLoadingIndicator(false);
        showAlert('Dimenzije uspješno izvezene u Excel. Preuzimanje započinje...', 'success');
        
        // Simuliraj preuzimanje (u pravoj implementaciji, ovo bi bio stvarni download)
        const dummyLink = document.createElement('a');
        dummyLink.href = '#';
        dummyLink.download = 'dimensions_export_' + new Date().toISOString().slice(0, 10) + '.xlsx';
        dummyLink.click();
    }, 1500);
}

/**
 * Resetira dokument i sve povezane podatke
 */
function resetDocument() {
    if (confirm('Jeste li sigurni da želite obrisati trenutni dokument i sve dimenzije?')) {
        // Resetiraj sve varijable
        currentFile = null;
        currentPdf = null;
        currentPage = 1;
        totalPages = 1;
        currentScale = 1.0;
        dimensions = [];
        isManualMode = false;
        
        // Resetiraj UI
        document.getElementById('uploadForm').reset();
        document.getElementById('viewerContainer').innerHTML = '';
        document.getElementById('viewerContainer').classList.add('d-none');
        document.getElementById('noDocumentMessage').classList.remove('d-none');
        
        // Onemogući gumbe
        disableControls();
        
        // Resetiraj ručni način
        const manualModeBtn = document.getElementById('manualModeBtn');
        manualModeBtn.classList.remove('btn-warning');
        manualModeBtn.classList.add('btn-info');
        manualModeBtn.innerHTML = '<i class="fas fa-pencil-alt me-2"></i>Ručno označavanje';
        document.querySelector('.document-viewer').classList.remove('manual-mode-active');
        
        // Očisti prikaz dimenzija
        renderDimensionsList();
        categorizeDimensions();
        
        showAlert('Dokument je uspješno obrisan.', 'success');
    }
}

/**
 * Zoom in funkcija
 */
function zoomIn() {
    currentScale += 0.2;
    if (currentPdf) {
        renderPage(currentPage);
    } else {
        const img = document.querySelector('#viewerContainer img');
        if (img) {
            img.style.transform = `scale(${currentScale})`;
        }
    }
}

/**
 * Zoom out funkcija
 */
function zoomOut() {
    if (currentScale > 0.4) {
        currentScale -= 0.2;
        if (currentPdf) {
            renderPage(currentPage);
        } else {
            const img = document.querySelector('#viewerContainer img');
            if (img) {
                img.style.transform = `scale(${currentScale})`;
            }
        }
    }
}

/**
 * Resetiraj zoom
 */
function resetZoom() {
    currentScale = 1.0;
    if (currentPdf) {
        renderPage(currentPage);
    } else {
        const img = document.querySelector('#viewerContainer img');
        if (img) {
            img.style.transform = `scale(${currentScale})`;
        }
    }
}

/**
 * Prikazuje/skriva loading indikator
 * @param {boolean} show - Treba li prikazati indikator
 */
function showLoadingIndicator(show) {
    const loadingIndicator = document.getElementById('loadingIndicator');
    if (show) {
        loadingIndicator.classList.remove('d-none');
    } else {
        loadingIndicator.classList.add('d-none');
    }
}

/**
 * Omogućuje kontrole nakon učitavanja dokumenta
 */
function enableControls() {
    document.getElementById('detectDimensionsBtn').disabled = false;
    document.getElementById('manualModeBtn').disabled = false;
    document.getElementById('exportBtn').disabled = false;
    document.getElementById('resetBtn').disabled = false;
    document.getElementById('zoomInBtn').disabled = false;
    document.getElementById('zoomOutBtn').disabled = false;
    document.getElementById('resetZoomBtn').disabled = false;
}

/**
 * Onemogućuje kontrole
 */
function disableControls() {
    document.getElementById('detectDimensionsBtn').disabled = true;
    document.getElementById('manualModeBtn').disabled = true;
    document.getElementById('exportBtn').disabled = true;
    document.getElementById('resetBtn').disabled = true;
    document.getElementById('zoomInBtn').disabled = true;
    document.getElementById('zoomOutBtn').disabled = true;
    document.getElementById('resetZoomBtn').disabled = true;
}

/**
 * Prikazuje obavijest korisniku
 * @param {string} message - Poruka za prikaz
 * @param {string} type - Tip obavijesti (success, warning, danger, info)
 */
function showAlert(message, type) {
    // Stvori element za obavijest
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    alertDiv.style.top = '20px';
    alertDiv.style.right = '20px';
    alertDiv.style.zIndex = '9999';
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Zatvori"></button>
    `;
    
    // Dodaj obavijest u DOM
    document.body.appendChild(alertDiv);
    
    // Automatski ukloni nakon 3 sekunde
    setTimeout(() => {
        alertDiv.classList.remove('show');
        setTimeout(() => {
            alertDiv.remove();
        }, 300);
    }, 3000);
}