<?php
/**
 * Inicijalizacija baze podataka
 * 
 * Datoteka: init_db.php
 * Opis: Stvara potrebne tablice u bazi podataka i inicijalizira direktorije
 */

// Definiranje osnovnih putanja za Linux hosting
$rootPath = '/var/www/html/';
$uploadsPath = $rootPath . 'uploads/';
$logsPath = $rootPath . 'logs/';

// Stvaranje potrebnih direktorija ako ne postoje
if (!file_exists($uploadsPath)) {
    mkdir($uploadsPath, 0755, true);
    echo "Stvoren uploads direktorij: $uploadsPath\n";
}

if (!file_exists($logsPath)) {
    mkdir($logsPath, 0755, true);
    echo "Stvoren logs direktorij: $logsPath\n";
}

// Uključivanje konfiguracije baze
require_once $rootPath . 'db_config.php';

// Ispisivanje informacija o procesu
echo "Inicijalizacija baze podataka mjerenjeproba...\n";

// Provjera postojanja baze
$checkDb = $conn->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = 'mjerenjeproba'");

if ($checkDb->num_rows == 0) {
    echo "Baza podataka ne postoji. Stvaranje baze podataka...\n";
    
    $conn->query("CREATE DATABASE IF NOT EXISTS mjerenjeproba CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    
    if ($conn->error) {
        die("Greška prilikom stvaranja baze podataka: " . $conn->error);
    }
    
    $conn->select_db('mjerenjeproba');
    echo "Baza podataka uspješno stvorena.\n";
} else {
    echo "Baza podataka već postoji.\n";
    $conn->select_db('mjerenjeproba');
}

// Stvaranje tablice za dokumente
$createDocumentsTable = "
CREATE TABLE IF NOT EXISTS documents (
    id INT AUTO_INCREMENT PRIMARY KEY,
    original_name VARCHAR(255) NOT NULL,
    file_path VARCHAR(255) NOT NULL COMMENT 'Putanja relativna prema uploads/',
    file_type VARCHAR(50) NOT NULL,
    upload_date DATETIME NOT NULL,
    marked_file_path VARCHAR(255) COMMENT 'Putanja relativna prema uploads/',
    excel_file_path VARCHAR(255) COMMENT 'Putanja relativna prema uploads/'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
";

if ($conn->query($createDocumentsTable)) {
    echo "Tablica 'documents' uspješno stvorena.\n";
} else {
    die("Greška prilikom stvaranja tablice 'documents': " . $conn->error);
}

// Stvaranje tablice za dimenzije
$createDimensionsTable = "
CREATE TABLE IF NOT EXISTS dimensions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    document_id INT NOT NULL,
    number INT NOT NULL,
    text VARCHAR(255) NOT NULL,
    type VARCHAR(50) NOT NULL,
    subtype VARCHAR(50),
    page INT DEFAULT 1,
    x0 FLOAT,
    y0 FLOAT,
    x1 FLOAT,
    y1 FLOAT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (document_id) REFERENCES documents(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
";

if ($conn->query($createDimensionsTable)) {
    echo "Tablica 'dimensions' uspješno stvorena.\n";
} else {
    die("Greška prilikom stvaranja tablice 'dimensions': " . $conn->error);
}

// Stvaranje tablice za kategorije
$createCategoriesTable = "
CREATE TABLE IF NOT EXISTS categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL,
    description TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
";

if ($conn->query($createCategoriesTable)) {
    echo "Tablica 'categories' uspješno stvorena.\n";
} else {
    die("Greška prilikom stvaranja tablice 'categories': " . $conn->error);
}

// Stvaranje tablice za podkategorije
$createSubcategoriesTable = "
CREATE TABLE IF NOT EXISTS subcategories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    name VARCHAR(50) NOT NULL,
    description TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
";

if ($conn->query($createSubcategoriesTable)) {
    echo "Tablica 'subcategories' uspješno stvorena.\n";
} else {
    die("Greška prilikom stvaranja tablice 'subcategories': " . $conn->error);
}

// Unos osnovnih kategorija
$categories = [
    ['Dimenzije', 'Osnovne dimenzije na tehničkom nacrtu'],
    ['Navoji', 'Specifikacije navoja'],
    ['Hrapavost', 'Oznake hrapavosti površina'],
    ['GD&T', 'Geometrijske tolerancije i dimenzioniranje']
];

$insertCategory = $conn->prepare("INSERT INTO categories (name, description) VALUES (?, ?)");

foreach ($categories as $category) {
    $insertCategory->bind_param("ss", $category[0], $category[1]);
    $insertCategory->execute();
    
    if ($insertCategory->error) {
        echo "Greška prilikom unosa kategorije '{$category[0]}': " . $insertCategory->error . "\n";
    } else {
        echo "Kategorija '{$category[0]}' uspješno unesena.\n";
    }
}

// Logiranje završetka inicijalizacije
$logMessage = date('Y-m-d H:i:s') . " - Database initialization completed\n";
file_put_contents($logsPath . 'db_init.log', $logMessage, FILE_APPEND);

echo "\nInicijalizacija baze podataka završena.\n";
echo "Log zapisan u: " . $logsPath . "db_init.log\n";

// Zatvaranje konekcije
$conn->close();